<?php
// models/User.php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

class User extends ActiveRecord implements IdentityInterface
{
    const SCENARIO_REGISTER = 'register';
    const SCENARIO_UPDATE = 'update';
    const SCENARIO_CHANGE_PASSWORD = 'change-password';

    public $password;
    public $password_repeat;
    public $agree_terms;
    public $new_password;
    public $confirm_password;

    public static function tableName()
    {
        return '{{%user}}';
    }

    public function rules()
    {
        return [
            // Обязательные поля для регистрации
            [['full_name', 'phone', 'email', 'address', 'residents_count'], 'required'],
            [['password', 'password_repeat'], 'required', 'on' => self::SCENARIO_REGISTER],
            ['agree_terms', 'required', 'requiredValue' => 1, 'message' => 'Вы должны согласиться с правилами регистрации', 'on' => self::SCENARIO_REGISTER],
            
            // Обязательные поля для смены пароля
            [['new_password', 'confirm_password'], 'required', 'on' => self::SCENARIO_CHANGE_PASSWORD],
            [['new_password'], 'string', 'min' => 10, 'message' => 'Пароль должен содержать минимум 10 символов', 'on' => self::SCENARIO_CHANGE_PASSWORD],
            ['confirm_password', 'compare', 'compareAttribute' => 'new_password', 'message' => 'Пароли не совпадают', 'on' => self::SCENARIO_CHANGE_PASSWORD],
            
            // Валидация форматов
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s\-]+$/u', 'message' => 'ФИО должно содержать только кириллические символы, пробелы и дефисы'],
            ['phone', 'match', 'pattern' => '/^8\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 'message' => 'Формат телефона: 8(XXX)XXX-XX-XX'],
            ['email', 'email', 'message' => 'Некорректный формат email'],
            ['residents_count', 'integer', 'min' => 1, 'message' => 'Количество проживающих должно быть целым числом больше 0'],
            
            // Валидация пароля для регистрации
            ['password', 'string', 'min' => 10, 'message' => 'Пароль должен содержать минимум 10 символов', 'on' => self::SCENARIO_REGISTER],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают', 'on' => self::SCENARIO_REGISTER],
            
            // Уникальность с исключением текущего пользователя
            ['phone', 'unique', 'targetClass' => self::class, 'message' => 'Этот телефон уже зарегистрирован', 'filter' => function($query) {
                if (!$this->isNewRecord) {
                    $query->andWhere(['not', ['id' => $this->id]]);
                }
            }],
            ['email', 'unique', 'targetClass' => self::class, 'message' => 'Этот email уже зарегистрирован', 'filter' => function($query) {
                if (!$this->isNewRecord) {
                    $query->andWhere(['not', ['id' => $this->id]]);
                }
            }],

            ['is_admin', 'boolean'],
        ];
    }

    public function scenarios()
    {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_REGISTER] = ['full_name', 'phone', 'email', 'address', 'residents_count', 'password', 'password_repeat', 'agree_terms'];
        $scenarios[self::SCENARIO_UPDATE] = ['full_name', 'phone', 'email', 'address', 'residents_count'];
        $scenarios[self::SCENARIO_CHANGE_PASSWORD] = ['new_password', 'confirm_password'];
        return $scenarios;
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($this->isNewRecord) {
                $this->auth_key = Yii::$app->security->generateRandomString();
                $this->is_admin = 0;
            }
            
            // Хеширование пароля при регистрации
            if ($this->isNewRecord && !empty($this->password)) {
                $this->setAttribute('password', Yii::$app->security->generatePasswordHash($this->password));
            }
            
            return true;
        }
        return false;
    }

    // IdentityInterface methods
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return static::findOne(['auth_key' => $token]);
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAuthKey()
    {
        return $this->auth_key;
    }

    public function validateAuthKey($authKey)
    {
        return $this->auth_key === $authKey;
    }

    public static function findByPhone($phone)
    {
        return static::findOne(['phone' => $phone]);
    }

    public function validatePassword($password)
    {
        $passwordHash = $this->getAttribute('password');
        if (empty($passwordHash)) {
            return false;
        }
        return Yii::$app->security->validatePassword($password, $passwordHash);
    }

    public function getCounterReadings()
    {
        return $this->hasMany(CounterReadings::class, ['user_id' => 'id']);
    }

    public function isAdmin()
    {
        return $this->is_admin == 1;
    }

    public function changePassword()
    {
        if ($this->new_password) {
            // Генерируем хеш нового пароля
            $passwordHash = Yii::$app->security->generatePasswordHash($this->new_password);
            
            // Сохраняем хеш в базу данных
            $this->password = $passwordHash;
            
            // Сохраняем только поле password
            return $this->updateAttributes(['password' => $passwordHash]);
        }
        return false;
    }

    public function attributeLabels()
    {
        return [
            'full_name' => 'ФИО',
            'phone' => 'Телефон',
            'email' => 'Email',
            'address' => 'Адрес проживания',
            'residents_count' => 'Количество проживающих',
            'password' => 'Пароль',
            'password_repeat' => 'Повтор пароля',
            'agree_terms' => 'Согласие с правилами регистрации',
            'new_password' => 'Новый пароль',
            'confirm_password' => 'Подтверждение пароля',
        ];
    }
}